<?php
/**
 * Class Api file.
 *
 * @package WebToffee\CookieConsent\Lite\Admin\Modules\Scanner\Api
 */

namespace WebToffee\CookieConsent\Lite\Admin\Modules\Scanner\Api;

use WP_REST_Server;
use WP_Error;
use stdClass;
use WebToffee\CookieConsent\Lite\Includes\Rest_Controller;
use WebToffee\CookieConsent\Lite\Admin\Modules\Scanner\Includes\Controller;
use WebToffee\CookieConsent\Lite\Admin\Modules\Scanner\Includes\Scanner;
use WebToffee\CookieConsent\Lite\Admin\Modules\Scanner\Includes\Cookie_Serve;
use Exception;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Cookies API
 *
 * @class       Api
 * @version     3.0.0
 * @package     WebToffee
 * @extends     Rest_Controller
 */
class Api extends Rest_Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wcc/v1';
	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'scans';

	protected $api = 'https://phplaravel-273836-1664986.cloudwaysapps.com';

	/**
	 * Scanner controller object.
	 *
	 * @var object
	 */
	protected $controller;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->controller = Controller::get_instance();
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 10 );
	}

	/**
	 * Register the routes for cookies.
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/bulk',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'bulk' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args' => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'webtoffee-cookie-consent' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::EDITABLE ),
				),
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::DELETABLE ),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/initiate',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'initiate_scan' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/start' . '/(?P<id>[\d]+)',
			array(
				'args' => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'webtoffee-cookie-consent' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'start_scan' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/abort' . '/(?P<id>[\d]+)',
			array(
				'args' => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'webtoffee-cookie-consent' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'abort' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/status' . '/(?P<id>[\d]+)',
			array(
				'args' => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'webtoffee-cookie-consent' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'scan_status' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/result' . '/(?P<id>[\d]+)',
			array(
				'args' => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'webtoffee-cookie-consent' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_scan_result' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param( array( 'default' => 'view' ) ),
					),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/ip',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'fetch_ip' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/preview',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'get_preview' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/presets',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_presets' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/configs',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_configs' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
		register_rest_route(
			'cookieyes/v1',
			'/fetch_results',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'fetch_result' ),
					'permission_callback' => '__return_true',
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}
	/**
	 * Get scan histories
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_items( $request ) {
		$history_details = array();
		$items           = $this->controller->get_item_from_db( $request );
		if ( isset( $items ) ) {
			foreach ( $items as $index => $item ) {
				$data                = new stdClass();
				$data->id            = isset( $item->id ) ? absint( $item->id ) : 0;
				$data->scan_status   = isset( $item->scan_status ) ? sanitize_text_field( $item->scan_status ) : '';
				$data->scan_id       = isset( $item->scan_id ) ? sanitize_text_field( $item->scan_id ) : '';
				$data->scan_token    = isset( $item->scan_token ) ? sanitize_text_field( $item->scan_token ) : '';
				$data->total_url     = isset( $item->total_url ) ? absint( $item->total_url ) : 0;
				$data->category      = isset( $item->category ) ? sanitize_text_field( $item->category ) : 0;
				$data->total_cookies = isset( $item->total_cookies ) ? absint( $item->total_cookies ) : 0;
				$data->total_scripts = isset( $item->total_scripts ) ? absint( $item->total_scripts ) : 0;
				$data->date_created  = isset( $item->date_created ) ? sanitize_text_field( $item->date_created ) : '';
				$data->urls_scanned  = isset( $item->urls_scanned ) ? sanitize_text_field( $item->urls_scanned ) : '';
				$data->scan_history  = isset( $item->scan_history ) ? sanitize_text_field( $item->scan_history ) : '';
				if ( ! empty( $data ) ) {
					$history_details['data'][ $index ] = $data;
				}
			}
		}
		return $history_details;
	}

	/**
	 * Get a collection of items.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_item( $request ) {
		$object = new Scanner( (int) $request['id'] );
		if ( ! $object || 0 === $object->get_id() ) {
			return new WP_Error( 'webtoffee_rest_invalid_id', __( 'Invalid ID.', 'webtoffee-cookie-consent' ), array( 'status' => 404 ) );
		}
		$data = $this->prepare_item_for_response( $object, $request );
		return rest_ensure_response( $data );
	}

	/**
	 * Create a new Scanner.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function create_item( $request ) {
		if ( ! empty( $request['id'] ) ) {
			return new WP_Error(
				'webtoffee_rest_item_exists',
				__( 'Cannot create existing Scanner.', 'webtoffee-cookie-consent' ),
				array( 'status' => 400 )
			);
		}
		$object = $this->prepare_item_for_database( $request );
		$object->save();
		$data = $this->prepare_item_for_response( $object, $request );
		return rest_ensure_response( $data );
	}

	/**
	 * Update an existing Scanner.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function update_item( $request ) {
		if ( empty( $request['id'] ) ) {
			return new WP_Error(
				'webtoffee_rest_item_exists',
				__( 'Invalid Scanner id', 'webtoffee-cookie-consent' ),
				array( 'status' => 400 )
			);
		}
		$registered = $this->get_collection_params();
		$object     = $this->prepare_item_for_database( $request );
		if ( isset( $registered['language'], $request['language'] ) ) {
			$object->set_language( sanitize_text_field( $request['language'] ) );
		}
		$object->save();
		$data = $this->prepare_item_for_response( $object, $request );
		return rest_ensure_response( $data );
	}

	/**
	 * Delete an existing Scanner.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function delete_item( $request ) {
		if ( empty( $request['id'] ) ) {
			return new WP_Error(
				'webtoffee_rest_item_exists',
				__( 'Invalid Scanner id', 'webtoffee-cookie-consent' ),
				array( 'status' => 400 )
			);
		}
		$scanner_id = $request['id'];
		$data       = $this->controller->remove( $scanner_id );
		return rest_ensure_response( $data );
	}

	/**
	 * Performs bulk update request.
	 *
	 * @param object $request WP request object.
	 * @return array
	 */
	public function bulk( $request ) {
		try {
			if ( ! isset( $request['Scanner'] ) ) {
				return new WP_Error( 'webtoffee_rest_invalid_data', __( 'No data specified to create/edit Scanner', 'webtoffee-cookie-consent' ), array( 'status' => 404 ) );
			}
			if ( ! defined( 'WCC_BULK_REQUEST' ) ) {
				define( 'WCC_BULK_REQUEST', true );
			}
			$item_objects = array();
			$objects      = array();
			$data         = $request['Scanner'];

			foreach ( $data as $_scanner ) {
				$object = $this->prepare_item_for_database( $_scanner );
				$object->save();
				$item_objects[] = $object;
			}
			foreach ( $item_objects as $data ) {
				$data      = $this->prepare_item_for_response( $data, $request );
				$objects[] = $this->prepare_response_for_collection( $data );
			}
			do_action( 'wcc_after_update_scanner' );
			return rest_ensure_response( $objects );
		} catch ( Exception $e ) {
			return new WP_Error( $e->getCode(), $e->getMessage(), array( 'status' => $e->getCode() ) );
		}
	}

	/**
	 * Initiate a scan.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function initiate_scan( $request ) {
		$controller = new Cookie_Serve();
		$response   = $controller->initiate_scan( $controller );
		if ( is_wp_error( $response ) ) {
			return rest_ensure_response( $response );
		}
		$object = $this->prepare_item_for_database( $response );
		$object->save();
		return rest_ensure_response( $response );
	}

	/**
	 * Start a scan.
	 * Connect with cookiyes scanner
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function start_scan( $request ) {
		$object = new Scanner( (int) $request['id'] );
		if ( ! $object || 0 === $object->get_id() ) {
			return new WP_Error( 'webtoffee_rest_invalid_id', __( 'Invalid ID.', 'webtoffee-cookie-consent' ), array( 'status' => 404 ) );
		}
		$controller = new Cookie_Serve();
		$response   = $controller->start_scan( $request, $controller );
		return rest_ensure_response( $response );
	}

	/**
	 * Abort a scan.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function abort( $request ) {
		$object = new Scanner( (int) $request['id'] );
		if ( ! $object || 0 === $object->get_id() ) {
			return new WP_Error( 'webtoffee_rest_invalid_id', __( 'Invalid ID.', 'webtoffee-cookie-consent' ), array( 'status' => 404 ) );
		}
		$controller = new Cookie_Serve();
		$response   = $controller->abort_scan( $request, $controller );
		$object     = $this->prepare_item_for_database( $response );
		$data       = $this->update_scan_status( $object );
		return rest_ensure_response( $response );
	}

	/**
	 * Fetch scan status.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function scan_status( $request ) {
		$object = new Scanner( (int) $request['id'] );
		if ( ! $object || 0 === $object->get_id() ) {
			return new WP_Error( 'webtoffee_rest_invalid_id', __( 'Invalid ID.', 'webtoffee-cookie-consent' ), array( 'status' => 404 ) );
		}
		$controller = new Cookie_Serve();
		$response   = $controller->fetch_scan_status( $request, $controller );
		$object     = $this->prepare_item_for_database( $response );
		$data       = $this->update_scan_status( $object );
		return rest_ensure_response( $response );
	}
	/**
	 * Fetch scan result.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_scan_result( $request ) {
		$object = new Scanner( (int) $request['id'] );
		if ( ! $object || 0 === $object->get_id() ) {
			return new WP_Error( 'webtoffee_rest_invalid_id', __( 'Invalid ID.', 'webtoffee-cookie-consent' ), array( 'status' => 404 ) );
		}
		$controller = new Cookie_Serve();
		$response   = $controller->fetch_scan_result( $request, $controller );
		$object     = $this->prepare_item_for_database( $response );
		$this->update_scan_result( $object );
		return rest_ensure_response( $response );
	}

	/**
	 * Fetch IP address.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function fetch_ip( $request ) {
		$ip_address = '';
		
		// Check for X-Forwarded-For header first
		if ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) && ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$ip_address = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
		} elseif ( isset( $_SERVER['REMOTE_ADDR'] ) && ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
			$ip_address = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}
		
		return rest_ensure_response( $ip_address );
	}
	/**
	 * Fetch scan results from scanner.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function fetch_result( $request ) {
		$response = $this->controller->update_scan_result( $request );
		$object   = $this->prepare_item_for_database( $response );
		$this->update_scan_result( $object );
		return rest_ensure_response( $response );
	}

	/**
	 * Update an existing scanner locally.
	 *
	 * @param object $scanner Scanner object.
	 * @return void
	 */
	public function update_scan_status( $scanner ) {
		global $wpdb;
		$data = array(
			'scan_status' => $scanner->get_scan_status(),
		);
		$wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery,WordPress.DB.PreparedSQL.NotPrepared
			$wpdb->prefix . 'wcc_cookie_scan_history',
			$data,
			array( 'scan_id' => $scanner->get_scan_id() ),
			array(
				'%s',
			)
		);
		if ( defined( 'WCC_BULK_REQUEST' ) && WCC_BULK_REQUEST ) {
			return;
		}
		do_action( 'wcc_after_update_scan_status' );
	}
	/**
	 * Update an existing scanner locally.
	 *
	 * @param object $scanner Scanner object.
	 * @return void
	 */
	public function update_scan_result( $scanner ) {
		global $wpdb;
		$data = array(
			'scan_status'   => $scanner->get_scan_status(),
			'total_url'     => $scanner->get_total_url(),
			'total_cookies' => $scanner->get_total_cookies(),
			'total_scripts' => $scanner->get_total_scripts(),
			'category'      => $scanner->get_category(),
			'urls_scanned'  => $scanner->get_urls_scanned(),
			'scan_history'  => $scanner->get_scan_history(),
		);
		$wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery,WordPress.DB.PreparedSQL.NotPrepared
			$wpdb->prefix . 'wcc_cookie_scan_history',
			$data,
			array( 'scan_id' => $scanner->get_scan_id() ),
			array(
				'%s',
				'%d',
				'%d',
				'%d',
				'%d',
				'%s',
				'%s',
			)
		);
		if ( defined( 'WCC_BULK_REQUEST' ) && WCC_BULK_REQUEST ) {
			return;
		}
		do_action( 'wcc_after_update_scan_status' );
	}

	/**
	 * Format data to provide output to API
	 *
	 * @param object $scanner scanner of the corresponding item.
	 * @param array  $request Request params.
	 * @return array
	 */
	public function prepare_item_for_response( $scanner, $request ) {
		$data    = $this->get_formatted_item_data( $scanner );
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );
		return rest_ensure_response( $data );
	}

	/**
	 * Format the support before sending.
	 *
	 * @param Scanner $scanner Scanner object.
	 * @return scanner
	 */
	public function get_formatted_item_data( $scanner ) {
		return array(
			'id'            => $scanner->get_id(),
			'scan_id'       => $scanner->get_scan_id(),
			'scan_status'   => $scanner->get_scan_status(),
			'scan_token'    => $scanner->get_scan_token(),
			'date_created'  => $scanner->get_date_created(),
			'total_url'     => $scanner->get_total_url(),
			'total_cookies' => $scanner->get_total_cookies(),
			'total_scripts' => $scanner->get_total_scripts(),
			'category'      => $scanner->get_category(),
			'urls_scanned'  => $scanner->get_urls_scanned(),
			'scan_history'  => $scanner->get_scan_history(),
		);
	}

	/**
	 * Prepare a single item for create or update.
	 *
	 * @param  WP_REST_Request $request Request object.
	 * @return object
	 */
	public function prepare_item_for_database( $request ) {
		$id     = isset( $request['id'] ) ? absint( $request['id'] ) : 0;
		$object = new Scanner( $id );
		$object->set_scan_id( isset( $request['scan_id'] ) ? $request['scan_id'] : '' );
		$object->set_scan_status( isset( $request['scan_status'] ) ? $request['scan_status'] : ( isset( $request['scan_result'] ) ? 'completed' : 'initiated' ) );
		$object->set_scan_token( isset( $request['scan_token'] ) ? $request['scan_token'] : '' );
		$object->set_date_created( isset( $request['date_created'] ) ? $request['date_created'] : '' );
		$object->set_total_url( $this->get_total_page_count() );
		$object->set_total_cookies( isset( $request['scan_result'] ) ? $this->get_total_cookie_count( $request['scan_result'] ) : '' );
		$object->set_total_scripts( isset( $request['scan_result'] ) ? $this->get_total_script_count( $request['scan_result'] ) : '' );
		$object->set_category( isset( $request['scan_result'] ) ? $this->get_total_category_count( $request['scan_result'] ) : '' );
		$object->set_urls_scanned( $this->get_scanned_urls() );
		$object->set_scan_history( isset( $request['scan_result'] ) ? $request['scan_result'] : '' );
		return $object;
	}
	/**
	 * Get the cookie count.
	 *
	 * @return integer
	 */
	public function get_total_cookie_count( $data_json ) {
		$total_cookies = 0;
		if ( ! empty( $data_json ) ) {
			// Parse the JSON data
			$data = json_decode( $data_json, true );
			foreach ( $data as $category ) {
				$total_cookies += count( $category['cookies'] );
			}
		}
		return $total_cookies;
	}
	/**
	 * Get the category count.
	 *
	 * @return integer
	 */
	public function get_total_category_count( $data_json ) {
		$length_of_array = 0;
		if ( ! empty( $data_json ) ) {
			// Parse the JSON data
			$data = json_decode( $data_json, true );
			// Get the length of the top-level array or object
			$length_of_array = count( $data );
		}
		return $length_of_array;
	}
	/**
	 * Get the script count.
	 *
	 * @return integer
	 */
	public function get_total_script_count( $data_json ) {
		$script_slug_count = 0;
		if ( ! empty( $data_json ) ) {
			// Parse the JSON data
			$data = json_decode( $data_json, true );
			foreach ( $data as $category ) {
				foreach ( $category['cookies'] as $cookie ) {
					if ( null !== $cookie['script_slug'] ) {
						++$script_slug_count;
					}
				}
			}
		}
		return $script_slug_count;
	}
	/**
	 * Get the total page count.
	 *
	 * @return integer
	 */
	public function get_total_page_count() {
		$length           = 0;
		$controller       = new Cookie_Serve();
		$get_urls_to_scan = $controller->get_pages();
		if ( isset( $get_urls_to_scan ) && $get_urls_to_scan['total'] ) {
			$length = $get_urls_to_scan['total'];
		}
		return $length;
	}

	/**
	 * Get the urls scanned
	 *
	 * @return array
	 */
	public function get_scanned_urls() {
		$urls_arr         = array();
		$controller       = new Cookie_Serve();
		$get_urls_to_scan = $controller->get_pages();
		if ( isset( $get_urls_to_scan ) && $get_urls_to_scan['log'] ) {
			$urls_arr = json_encode( $get_urls_to_scan['log'] );
		}
		return $urls_arr;
	}
	/**
	 * Get the query params for collections.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		return array(
			'context'  => $this->get_context_param( array( 'default' => 'view' ) ),
			'search'   => array(
				'description'       => __( 'Limit results to those matching a string.', 'webtoffee-cookie-consent' ),
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'validate_callback' => 'rest_validate_request_arg',
			),
			'ver'      => array(
				'description'       => __( 'Version', 'webtoffee-cookie-consent' ),
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'validate_callback' => 'rest_validate_request_arg',
			),
			'language' => array(
				'description'       => __( 'Language of the Scanner', 'webtoffee-cookie-consent' ),
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'validate_callback' => 'rest_validate_request_arg',
			),

		);
	}

	/**
	 * Get the Consent logs's schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'Scanner',
			'type'       => 'object',
			'properties' => array(
				'id'            => array(
					'description' => __( 'Unique identifier for the resource.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view' ),
					'readonly'    => true,
				),
				'scan_id'       => array(
					'description' => __( 'Scan id unique for each scan.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
				'scan_status'   => array(
					'description' => __( 'Status of current scan', 'webtoffee-cookie-consent' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'scan_token'    => array(
					'description' => __( 'Scan token unique for each scan', 'webtoffee-cookie-consent' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'date_created'  => array(
					'description' => __( 'The date the scan was created, as GMT.', 'webtoffee-cookie-consent' ),
					'type'        => 'date-time',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'total_url'     => array(
					'description' => __( 'No of URL\'s scanned.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
				'total_cookies' => array(
					'description' => __( 'No of cookies found.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
				'category'      => array(
					'description' => __( 'No of categories found.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
				'total_scripts' => array(
					'description' => __( 'No of scripts found.', 'webtoffee-cookie-consent' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
				'urls_scanned'  => array(
					'description' => __( 'List of URL\'s scanned.', 'webtoffee-cookie-consent' ),
					'type'        => 'array',
					'context'     => array( 'view', 'edit' ),
				),
				'scan_history'  => array(
					'description' => __( 'Result of the scan', 'webtoffee-cookie-consent' ),
					'type'        => 'object',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		return $this->add_additional_fields_schema( $schema );
	}
} // End the class.
